﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace VA.PPMS.IWS.Common.Extensions
{
    public static class DateTimeExtensions
    {
        public static DateTime? ToCrmDate(this DateTime dt)
        {
            return new DateTime(dt.Year, dt.Month, dt.Day, dt.Hour, dt.Minute, dt.Second, DateTimeKind.Local);
        }

        public static DateTime StartOfWeek(this DateTime dt, DayOfWeek startOfWeek)
        {
            var delta = startOfWeek - dt.DayOfWeek;
            if (delta > 0) delta -= 7;

            return dt.AddDays(delta);
        }

        public static bool IsAfterHours(this DateTime dt)
        {
            return (dt.Hour / 4) < 1;
        }

        public static bool IsAfterHoursWeekend(this DateTime dt)
        {
            return (dt.DayOfWeek == DayOfWeek.Sunday && dt.IsAfterHours());
        }

        public static bool IsProcessingWindow(this DateTime dt)
        {
            if (dt.DayOfWeek == DayOfWeek.Sunday || dt.DayOfWeek == DayOfWeek.Saturday)
            {
                return true;
            }
            else
            {
                var utcDt = dt.ToUniversalTime();
                int hour = utcDt.Hour;
                if (hour >= 1 && hour < 10)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Determine if the current time is within the allowed provider payload processing window
        /// </summary>
        /// <param name="dt">The date</param>
        /// <param name="startHour">Optional, hour processing starts</param>
        /// <param name="endHour">Optional, hour processing ends</param>
        /// <returns>true, if the date/time is within allowed processing window</returns>
        public static bool AllowProviderPayloadProcessing(this DateTime dt, int startHour, int endHour, bool ignoreHolidays = false)
        {
            // Determine if date is a holiday
            bool isHoliday = false;
            if (!ignoreHolidays)
            {
                DateTime dateOnly = new DateTime(dt.Year, dt.Month, dt.Day);
                IEnumerable<DateTime> holidays = VaHolidays.GetHolidays(dt.Year);
                isHoliday = holidays.Any(x => x.Date == dateOnly);
            }

            // If holiday or weekend, process
            if (dt.DayOfWeek == DayOfWeek.Sunday || dt.DayOfWeek == DayOfWeek.Saturday || isHoliday)
            {
                return true;
            }
            else
            {
                // If within time window, process
                var utcDt = dt.ToUniversalTime();
                int hour = utcDt.Hour;

                // Check for DST
                if (dt.IsDaylightSavingTime()) hour = FallBack(hour, 1);

                var start = TimeSpan.Parse($"{startHour}:00");
                var end = TimeSpan.Parse($"{endHour}:00");
                var now = TimeSpan.Parse($"{hour}:00");

                if (start <= end)
                {
                    return now >= start && now < end;
                }
                else
                {
                    return now >= start || now < end;
                }
            }
        }

        public static int FallBack(int hour, int offset)
        {
            var ts = new TimeSpan(hour, 0, 0);
            ts = ts.Subtract(new TimeSpan(offset, 0, 0));
            return ts.Hours;
        }
    }
}
